// Adobe.FaceTracker.js
/*global define */
/*jslint sub: true */  

define([ "lib/Zoot", "src/utils", "src/math/Vec2", "src/math/mathUtils", "lib/dev", "lib/tasks", "lodash", "src/animate/faceUtils" ],
  function (Zoot, utils, v2, mathUtils, dev, tasks, lodash, faceUtils) {
		"use strict";

	// mapping from keyboard inputs to eye gaze offsets
	var	kc = Zoot.keyCodes,

		aKeyCodeOffsets = [
			// Note: multi-key combinations need to come before single-keys bc we trigger the first match found in this array
			// The offset values are set based on the assumption that up/down motion should be slightly less (0.8) than left/right motion (1.0).
			// This forms an ellipse with major axis 1.0 and minor axis 0.8. The offsets below represent points on this ellipse at the 4 cardinal
			// directions and the 4 "corners" (45 degrees). This is all just an approximation since we aren't taking into account the actual eyeball
			// and eyelid geometries, but it seems to work reasonably well for many examples.
			{ "keyCodes" : [kc.leftKey, kc.upKey], 		"offset" : faceUtils.getEyeDartDirectionFromAngle(1.25 * Math.PI) },
			{ "keyCodes" : [kc.rightKey, kc.upKey],		"offset" : faceUtils.getEyeDartDirectionFromAngle(1.75 * Math.PI) },
			{ "keyCodes" : [kc.leftKey, kc.downKey],	"offset" : faceUtils.getEyeDartDirectionFromAngle(0.75 * Math.PI) },
			{ "keyCodes" : [kc.rightKey, kc.downKey],	"offset" : faceUtils.getEyeDartDirectionFromAngle(0.25 * Math.PI)},
			{ "keyCodes" : [kc.leftKey],				"offset" : faceUtils.getEyeDartDirectionFromAngle(Math.PI) },
			{ "keyCodes" : [kc.rightKey],				"offset" : faceUtils.getEyeDartDirectionFromAngle(0) },
			{ "keyCodes" : [kc.upKey], 					"offset" : faceUtils.getEyeDartDirectionFromAngle(1.5 * Math.PI) },
			{ "keyCodes" : [kc.downKey],				"offset" : faceUtils.getEyeDartDirectionFromAngle(0.5 * Math.PI) },
		];

	// compute mouse eye gaze offset
	function computeNormalizedMouseEyeGazeOffset(args) {
		var	mouseVec, mouseEyeGazeOffset = [0, 0], 
			deltaFromSceneCenter, distFromSceneCenter, angle, 			
			epsilon = 0.00001,
            paramInput = "mouseEyeGaze",
            leftDownB = args.getParamEventValue("mouseEyeGaze", "Mouse/Down/Left", null, 0, true),
            mousePosition0 = args.getParamEventValue("mouseEyeGaze", "Mouse/Position", null, [0, 0], true),
            touchEvents, touchId, touchIdKey;

        // if there are touch events, use that instead of the mouse
		touchEvents = args.getParamEventValue(paramInput, "Touch/IndexToID/Count", null, 0, true);
		if (touchEvents) {
			touchId = args.getParamEventValue(paramInput, "Touch/IndexToID/0", null, 0, true);
			touchIdKey = "Touch/ID/" + touchId;
			leftDownB = args.getParamEventValue(paramInput, touchIdKey + "/Down", null, 0, true);
			if (leftDownB) {
				mousePosition0 = args.getParamEventValue(paramInput, touchIdKey + "/Position", null, [0,0], true);
			}
		}
        
        if (leftDownB) {
			if (mousePosition0) {
				mouseVec = v2(mousePosition0);
				args.setEventGraphParamRecordingValid(paramInput);

				// eye gaze determined by mouse position wrt a circle with 200px radius
				// at the center of the scene. the perimeter of the circle represents 
				// the boundary of the eye.

				deltaFromSceneCenter = mouseVec;
				distFromSceneCenter = v2.magnitude(deltaFromSceneCenter);
				deltaFromSceneCenter = v2.normalize(deltaFromSceneCenter, v2());

				// note: Math.atan seems to return a valid result (PI/2) even if you 
				// call it with NaN (e.g., 1/0), but just to be safe, we're checking
				// for deltaFromSceneCenter[0] > epsilon here and explicitly setting the angle. -wilmotli
				if (deltaFromSceneCenter[0] > epsilon) {
					angle = Math.atan(deltaFromSceneCenter[1]/deltaFromSceneCenter[0]); 
				} else if (deltaFromSceneCenter[0] < -epsilon) {
					angle = -Math.PI + Math.atan(deltaFromSceneCenter[1]/deltaFromSceneCenter[0]);
				} else {
					angle = (deltaFromSceneCenter[1] > 0) ? 0.5 * Math.PI : -0.5 * Math.PI;
				}

				// clamp to a 200px radius circle
				distFromSceneCenter /= 200;
				if (distFromSceneCenter > 1) distFromSceneCenter = 1;

				// compute offset vector
				mouseEyeGazeOffset[0] = distFromSceneCenter * Math.cos(angle);
				mouseEyeGazeOffset[1] = distFromSceneCenter * Math.sin(angle);
			}
		}

		return mouseEyeGazeOffset;
	}	

	// compute keyboard eye gaze offset
	function computeNormalizedKeyboardEyeGazeOffset(args) {
		var kc = Zoot.keyCodes, keyboardEyeGazeOffset = [0, 0], 
			numKeyCodeOffsets = aKeyCodeOffsets.length, 
			numKeyCodes, keyCodes, keyCode, keyDown, allKeysDown, offset;

		for (var i=0; i<numKeyCodeOffsets; ++i) {
			keyCodes = aKeyCodeOffsets[i].keyCodes;
			offset = aKeyCodeOffsets[i].offset;
			numKeyCodes = keyCodes.length;

			allKeysDown = true;
			for (var j=0; j<numKeyCodes; ++j) {
				keyCode = keyCodes[j];
				keyDown = args.getParamEventValue("keyboardEyeGaze", kc.getKeyGraphId(keyCode), null, 0, true);	
				if (!keyDown) {
					allKeysDown = false;
					break;
				}			
			}

			if (allKeysDown) {
				keyboardEyeGazeOffset = offset;
				args.setEventGraphParamRecordingValid("keyboardEyeGaze");
				break;
			}
		}

		return keyboardEyeGazeOffset;
	}
	
	function getEyeGazeOffset(args, inputParamId, scaleFactor) {
		var offset = [0, 0], paramOutputKey = args.getParamEventOutputKey(inputParamId);

		offset[0] = args.getParamEventValue(inputParamId, paramOutputKey + "EyeGazeOffset/X", null, 0, true) || 0;
		offset[1] = args.getParamEventValue(inputParamId, paramOutputKey + "EyeGazeOffset/Y", null, 0, true) || 0;
	
		offset = v2.scale(scaleFactor, offset);

		return offset;	
	}

	function computeCombinedMouseAndKeyboardEyeGazeOffset(self, args, viewIndex, leftOrRight, normalizedMouseEyeGazeOffset, normalizedKeyboardEyeGazeOffset) {
		// add mouse and keyboard based eye gaze
		var eyeGazeRangeX = faceUtils.getPuppetMeasurement(self, leftOrRight + "EyeGazeRangeX", args, viewIndex),
			eyeGazeRangeY = faceUtils.getPuppetMeasurement(self, leftOrRight + "EyeGazeRangeY", args, viewIndex),
			mouseEyeGazeOffset = v2.initWithEntries(normalizedMouseEyeGazeOffset[0] * eyeGazeRangeX, normalizedMouseEyeGazeOffset[1] * eyeGazeRangeY),
			keyboardEyeGazeOffset = v2.initWithEntries(normalizedKeyboardEyeGazeOffset[0] * eyeGazeRangeX, normalizedKeyboardEyeGazeOffset[1] * eyeGazeRangeY),
			mouseAndKeyboardEyeGazeOffset = v2.add(mouseEyeGazeOffset, keyboardEyeGazeOffset);

		return mouseAndKeyboardEyeGazeOffset;
	}

	function computePuppetTransformsForEyeGaze(self, args, head14, viewIndex) {
		var faceTransforms = faceUtils.getPuppetTransforms(self, args, head14, viewIndex), transforms = [];
		
		transforms["Adobe.Face.LeftPupil"] = faceTransforms["Adobe.Face.LeftPupil"];
		transforms["Adobe.Face.RightPupil"] = faceTransforms["Adobe.Face.RightPupil"];

		// adjust camera eye gaze offset
		faceUtils.applyParamFactorToNamedTransformCustom(self, "eyeGazeFactor",	args, transforms, "Adobe.Face.LeftPupil", { pos : 1, scale : 0, rot : 0 });
		faceUtils.applyParamFactorToNamedTransformCustom(self, "eyeGazeFactor",	args, transforms, "Adobe.Face.RightPupil", { pos : 1, scale : 0, rot : 0 });

		// compute adjusted mouse and keyboard offsets
		var mouseEyeGazeFactor = args.getParam("mouseEyeGazeFactor") / 100,
			keyboardEyeGazeFactor = args.getParam("keyboardEyeGazeFactor") / 100,
			normalizedMouseEyeGazeOffset = getEyeGazeOffset(args, "mouseEyeGaze", mouseEyeGazeFactor),
			normalizedKeyboardEyeGazeOffset = getEyeGazeOffset(args, "keyboardEyeGaze", keyboardEyeGazeFactor);

		// add mouse and keyboard offsets to camera offset for left/right eyes			
		if (transforms["Adobe.Face.LeftPupil"]) {
			var leftMouseAndKeyboardEyeGazeOffset = computeCombinedMouseAndKeyboardEyeGazeOffset(self, args, viewIndex, "left", normalizedMouseEyeGazeOffset, normalizedKeyboardEyeGazeOffset),
				leftEyeGazeOffset = v2.add(transforms["Adobe.Face.LeftPupil"].translate, leftMouseAndKeyboardEyeGazeOffset);

			transforms["Adobe.Face.LeftPupil"].translate = leftEyeGazeOffset;
		}
		if (transforms["Adobe.Face.RightPupil"]) {
			var rightMouseAndKeyboardEyeGazeOffset = computeCombinedMouseAndKeyboardEyeGazeOffset(self, args, viewIndex, "right", normalizedMouseEyeGazeOffset, normalizedKeyboardEyeGazeOffset),
				rightEyeGazeOffset = v2.add(transforms["Adobe.Face.RightPupil"].translate, rightMouseAndKeyboardEyeGazeOffset);

			transforms["Adobe.Face.RightPupil"].translate = rightEyeGazeOffset;
		}

		return transforms;
	}	
	
	function animateWithFaceTracker(self, args) {
		var head14 = faceUtils.getFilteredHead14(args);

		self.aViews.forEach(function (view, viewIndex) {		
			var transforms = computePuppetTransformsForEyeGaze(self, args, head14, viewIndex);
			
			faceUtils.applyPuppetTransforms(self, args, viewIndex, transforms);
		});
	}
	
	function initMouseAndKeyboardEyeGazeSmoothingFilters(self) {
		self.mouseEyeGazeSmoothingFilter = new faceUtils.SmoothingFilter();
		self.keyboardEyeGazeSmoothingFilter = new faceUtils.SmoothingFilter();
	}

	// note: code below duplicates some of the logic in faceUtils.onFilterLiveInputs; could potentially be refactored ...
	function publishFilteredEyeGazeOffsetsFromInput(self, args, computeOffset, inputParamId, filter, filterParams) {
		var offset = computeOffset(args), inputLiveB = args.isParamEventLive(inputParamId), paramOutputKey = args.getParamEventOutputKey(inputParamId);

		if (inputLiveB) {
			var smoothingRate = (filterParams.smoothingRate !== undefined) ? filterParams.smoothingRate : 0, 
				snapEyeGazeEnabledB = (filterParams.snapEyeGazeEnabledB !== undefined) ? filterParams.snapEyeGazeEnabledB : false,
				eyeDartParams = {
					minGazeDuration: (filterParams.minGazeDuration !== undefined) ? filterParams.minGazeDuration : 0,
					numEyeDartDirections: (filterParams.numEyeDartDirections !== undefined) ? filterParams.numEyeDartDirections : 8,
					minSnapOffsetDist: (filterParams.minSnapOffsetDist !== undefined) ? filterParams.minSnapOffsetDist : 0.5,
					leftRightScaleFactor: (filterParams.leftRightScaleFactor !== undefined) ? filterParams.leftRightScaleFactor : 0.7,		
					upDownScaleFactor: (filterParams.upDownScaleFactor !== undefined) ? filterParams.upDownScaleFactor : 0.8,		
					offAxisScaleFactor: (filterParams.offAxisScaleFactor !== undefined) ? filterParams.offAxisScaleFactor : 1
				},
				pauseKeyDownB = faceUtils.getPauseKeyValue(inputParamId, args); // WL: should we use true/false instead? was 0/1 before ...

			if (pauseKeyDownB) {
				pauseKeyDownB = 1;
			}

			// no need to snap offsets set by keyboard input
			if (snapEyeGazeEnabledB && inputParamId !== "keyboardEyeGaze") {
				offset = faceUtils.snapToPredefinedEyeDartDirection(self, args.currentTime, eyeDartParams, offset);
			}

			if (filter) {
				offset = filter.getData(args.currentTime, smoothingRate, offset, pauseKeyDownB);
			}

			var sustainPreviousValue = (snapEyeGazeEnabledB && smoothingRate === 0);
			args.eventGraph.publish1D(paramOutputKey + "EyeGazeOffset/X", args.currentTime, offset[0], sustainPreviousValue);
			args.eventGraph.publish1D(paramOutputKey + "EyeGazeOffset/Y", args.currentTime, offset[1], sustainPreviousValue);	
		}	
	}
	
	var cameraInputParamDef = faceUtils.cameraInputParameterDefinitionV2; 
	
	cameraInputParamDef.uiToolTip = "$$$/animal/Behavior/EyeGaze/param/cameraInput/tooltip=Control eye gaze direction via your webcam; pause the eye gaze by holding down semicolon (;)";
	
	return {
		about:			"$$$/private/animal/Behavior/EyeGaze/About=Eye Gaze, (c) 2016.",
		description:	"$$$/animal/Behavior/EyeGaze/Desc=Controls pupil movement",
		uiName:			"$$$/animal/Behavior/EyeGaze/UIName=Eye Gaze",
		defaultArmedForRecordOn: true,
	
		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
		  return [
			cameraInputParamDef,
			{ id: "mouseEyeGaze", type: "eventGraph", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/mouseEyeGaze=Mouse & Touch Input",
				inputKeysArray: ["Mouse/", "Touch/", faceUtils.cameraInputPauseKeyCodeEnglish, faceUtils.cameraInputPauseKeyCodeGerman],
				outputKeyTraits: {
					takeGroupsArray: [
						{
							id: "EyeGazeOffset/*"
						}
					]
				},
			 	supportsBlending: true,
				uiToolTip: "$$$/animal/behavior/EyeGaze/param/mouseEyeGaze/tooltip=Control eye gaze direction with mouse; pause the eye gaze by holding down semicolon (;)" },
			{ id: "keyboardEyeGaze", type: "eventGraph", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/keyboardEyeGaze=Keyboard Input",
				inputKeysArray: ["Keyboard/", faceUtils.cameraInputPauseKeyCodeEnglish, faceUtils.cameraInputPauseKeyCodeGerman],
				outputKeyTraits: {
					takeGroupsArray: [
						{
							id: "EyeGazeOffset/*"
						}
					]
				},
			 	supportsBlending: true,
				uiToolTip: "$$$/animal/behavior/EyeGaze/param/keyboardEyeGaze/tooltip=Control eye gaze direction with the Arrow keys; pause the eye gaze by holding down semicolon (;)" },				
			{ id: "smoothingRate", type: "slider", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/smoothingRate=Smoothing", uiUnits: "%", min: 0, max: 100, precision: 0, dephault: 15, hideRecordButton: true,
				uiToolTip: "$$$/animal/behavior/EyeGaze/param/smoothingRate/tooltip=Exaggerate or minimize the smoothness when transitioning between different pupil positions" },
			{ id: "eyeGazeFactor", type: "slider", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/eyeGazeFactor=Camera Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/EyeGaze/param/eyeGazeFactor/tooltip=Exaggerate or minimize how far the pupils move when you look around" },
			{ id: "mouseEyeGazeFactor", type: "slider", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/mouseEyeGazeFactor=Mouse & Touch Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/EyeGaze/param/mouseEyeGazeFactor/tooltip=Exaggerate or minimize how far the pupils move when you drag using the mouse or touch screen" },
			{ id: "keyboardEyeGazeFactor", type: "slider", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/keyboardEyeGazeFactor=Keyboard Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/EyeGaze/param/keyboardEyeGazeFactor/tooltip=Exaggerate or minimize how far the pupils move when you use the Arrow keys" },				
			{ id: "snapEyeGaze", type: "checkbox", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/snapEyeGaze=Snap Eye Gaze", dephault: true, hideRecordButton: true,
				uiToolTip: "$$$/animal/Behavior/EyeGaze/Parameter/snapEyeGaze/tooltip=Snap eye gaze to predefined directions controlled by camera or mouse" },
			{ id: "minGazeDuration", type: "slider", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/minGazeDuration=Minimum Snap Duration", uiUnits: "$$$/animal/InspectScene/units/sec=sec", min: 0, max: 10, precision: 1, dephault: 1, hideRecordButton: true,
				uiToolTip: "$$$/animal/Behavior/EyeGaze/Parameter/minGazeDuration/tooltip=Limits how quickly eye gaze snapping happens" },
			{ id: "numEyeDartDirections", type: "slider", uiName: "$$$/private/animal/Behavior/EyeGaze/Parameter/numEyeDartDirections=Number of Eye Directions", min: 0, max: 16, precision: 0, dephault: 8, hidden: true,
				uiToolTip: "$$$/private/animal/Behavior/EyeGaze/Parameter/numEyeDartDirections/tooltip=Set number of eye dart directions to snap to" },
			{ id: "minSnapOffsetDist", type: "slider", uiName: "$$$/private/animal/Behavior/EyeGaze/Parameter/minSnapOffsetDist=Minimum Dist for Snapping", min: 0, max: 1, precision: 2, dephault: 0.25, hidden: true,
				uiToolTip: "$$$/private/animal/Behavior/EyeGaze/Parameter/minSnapOffsetDist/tooltip=Set minimum offset distance from center to trigger snapping" },				
			{ id: "leftRightScaleFactor", type: "slider", uiName: "$$$/private/animal/Behavior/EyeGaze/Parameter/leftRightScaleFactor=Left-Right Dart Sensitivity", min: 0, precision: 1, dephault: 0.7, hidden: true,
				uiToolTip: "$$$/private/animal/Behavior/EyeGaze/Parameter/leftRightScaleFactor/tooltip=Scale sensitivity of left-right eye gaze offset (lower means more likely to choose left-right directions)" },				
			{ id: "upDownScaleFactor", type: "slider", uiName: "$$$/private/animal/Behavior/EyeGaze/Parameter/upDownScaleFactor=Up-Down Dart Sensitivity", min: 0, precision: 1, dephault: 0.8, hidden: true,
				uiToolTip: "$$$/private/animal/Behavior/EyeGaze/Parameter/upDownScaleFactor/tooltip=Scale sensitivity of up-down eye gaze offset (lower means more likely to choose up-down directions)" },				
			{ id: "offAxisScaleFactor", type: "slider", uiName: "$$$/private/animal/Behavior/EyeGaze/Parameter/offAxisScaleFactor=Off-axis Dart Sensitivity", min: 0, precision: 1, dephault: 1, hidden: true,
				uiToolTip: "$$$/private/animal/Behavior/EyeGaze/Parameter/offAxisScaleFactor/tooltip=Scale sensitivity of off-axis eye gaze offset (lower means more likely to choose off-axis directions)" },				
			{ id: "viewLayers", type: "layer", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/Views=Views", dephault: { match:["//Adobe.Face.LeftProfile|Adobe.Face.LeftQuarter|Adobe.Face.Front|Adobe.Face.RightQuarter|Adobe.Face.RightProfile|Adobe.Face.Upward|Adobe.Face.Downward", "."] } },
			{ id: "handlesGroup", type: "group", uiName: "$$$/animal/Behavior/EyeGaze/Parameter/handlesGroup=Handles", groupChildren: faceUtils.defineHandleParams(true, false) }
		  ];
		},
		
		defineTags: function () {
			var aAllTags = [];
			var aAllTagRefs = [faceUtils.headFeatureTagDefinitions, faceUtils.eyeFeatureTagDefinitions, faceUtils.pupilFeatureTagDefinitions, faceUtils.eyelidLayerTagDefinitions, faceUtils.pupilLayerTagDefinitions/*, faceUtils.mouthShapeLayerTagDefinitions, faceUtils.mouthParentLayerTagDefinition*/, faceUtils.viewLayerTagDefinitions];
			aAllTagRefs.forEach(function (ar) {
				aAllTags = aAllTags.concat(ar);
			}); 
			return {aTags:aAllTags};
		},
		
		onCreateBackStageBehavior: function (/*self*/) {
			return { order: 0.52, importance : 0.0 }; // must come after LipSync
		},
		
		onCreateStageBehavior: function (self, args) {
			faceUtils.onCreateStageBehavior(self, args, false, true, false);
			initMouseAndKeyboardEyeGazeSmoothingFilters(self);
			self.lastEyeDartTime = null;
			self.lastEyeDartOffset = v2.initWithEntries(0.0, 0.0);
		},

		// Clear the rehearsal state
		onResetRehearsalData : function (self) {
			faceUtils.onResetRehearsalData(self);
			initMouseAndKeyboardEyeGazeSmoothingFilters(self);
			self.lastEyeDartTime = null;
			self.lastEyeDartOffset = v2.initWithEntries(0.0, 0.0);
		},

		onFilterLiveInputs: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			faceUtils.onFilterLiveInputs(self, args, /*filterEyeGazeB*/ true);
			
			var mouseInputParamId = "mouseEyeGaze",
				keyboardInputParamId = "keyboardEyeGaze",
				filterParams = {
					smoothingRate : args.getParam("smoothingRate"), 
					snapEyeGazeEnabledB: args.getParam("snapEyeGaze"),
					minGazeDuration: args.getParam("minGazeDuration"),
					numEyeDartDirections : args.getParam("numEyeDartDirections"),
					minSnapOffsetDist : args.getParam("minSnapOffsetDist"),
					leftRightScaleFactor : args.getParam("leftRightScaleFactor"),
					upDownScaleFactor : args.getParam("upDownScaleFactor"),
					offAxisScaleFactor : args.getParam("offAxisScaleFactor")
				};

			publishFilteredEyeGazeOffsetsFromInput(self, args, computeNormalizedKeyboardEyeGazeOffset, keyboardInputParamId, self.keyboardEyeGazeSmoothingFilter, filterParams);
			publishFilteredEyeGazeOffsetsFromInput(self, args, computeNormalizedMouseEyeGazeOffset, mouseInputParamId, self.mouseEyeGazeSmoothingFilter, filterParams);

		},
		
		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			animateWithFaceTracker(self, args);
		}
		
	}; // end of object being returned
});
